#!/usr/bin/env python3

##    MIT License
##
##    Copyright (c) 2025 and later AJ_Lethal
##
##    Permission is hereby granted, free of charge, to any person obtaining a copy
##    of this software and associated documentation files (the "Software"), to deal
##    in the Software without restriction, including without limitation the rights
##    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
##    copies of the Software, and to permit persons to whom the Software is
##    furnished to do so, subject to the following conditions:
##
##    The above copyright notice and this permission notice shall be included in all
##    copies or substantial portions of the Software.
##
##    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
##    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
##    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
##    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
##    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
##    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
##    SOFTWARE.

import tkinter as tk
from tkinter import filedialog
from tkinter import messagebox
from tkinter import ttk
import os
import io
import binascii
import struct
import json
import hashlib
import shutil
import datetime
from idlelib.tooltip import Hovertip

## sets current working directory to script location's
os.chdir(os.path.realpath(os.path.dirname(__file__)))

## "About" dialog
def aboutDlg(*args):
    aboutMsg = messagebox.showinfo("About BusyKip",
                                   "BusyKip v1.0 \n"
                                   "A tool to tag and export NFSMW Unlimiter Presitter presets.\n"
                                    "(c) 2025 and later AJ_Lethal\n\n"
                                    "Licensed under the MIT License")

## icon data
sIconData = b'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAB/ElEQVQ4y8WTT0gUcRTHP7+Z38zOOrs22JZ/UwM33Qy0W4pEh0CCTkXgdYXCCKRrHSW6ltcOeegQQXQWugh1E6wOicm2gcaioq4xO7vuzp9fB5dUSgk69G7vz+f7Ho/34B9NHJfsabUHgBvHlMyIY+DnQDYR00glJQCaAFXPr+34VHyl5BHwbSCbaYvxJJsmHtMIwr1xlQBTCj58dZmcyQt5AOoGBuvu4zZH8nQ8TffpOA9e5HgzX/zVYOJqM0N9JwCQdfAlcMkyBElLB+Di2QSGLrj3bImFb+UjlyCBvGVqYrDT4v71M5i6jh9GFLarBBHMLZYIIkVzo2RqrItyNWK5UD4kIB6NdZLpaKCwXaO33UIIKHo+ywWPobTNuy8lhnsTXLnQRBAqNn7UDgnQGNdZXC3z6v0GqaRk0w0439HA3WsdxE2dTXeF/PouO15A1Y/Y9SNQal9AQyO35tLfZauuUxZLK564M9pOU8JgJOPQ197A4qqHY+/t/NZwM8sFb1+gGoRoQItjiFTSUCMZh9mFLS73O+TXKmy5Po4tqdQi4qbGyaSBqN+gBNTc56IYHUhRDULmcy6RQJkawtsNmf24zXqxxsOb3Zjy97sTPa12CbAPxBQQnmuJyenxNIbUAIUpNcQB/u2nLaZef//zL/S02tPA5F/8kuK/208iyqeHAKlM3wAAAABJRU5ErkJggg=='
lIconData = b'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'

## button icon data, icons based on Tango Desktop set
openBtnIconData = b'iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAAACXBIWXMAAA3WAAAN1gGQb3mcAAAD0ElEQVRIx6WVS2tdVRTHf3vvcx9NbrANgkI7SgLFWsWUDgwU4geooyY4FIuIH8CB4sB+BRUc9AOI0IHFgQOhLQFrrXYifVCtVGNI2t60zc19nHPu2Xuv5eA+cpNg29g/LNicc9b67b0e+xieosV3Tp2x1nwCxqqKMcb03/QW5VLprkp8t9uVa+fOnSt2+huAxcVFNzU1Vcvz3Oz84EH93qcnTpz46PR772OM2WYA589/y4VLF9aajcbpiYn9F8+ePetH/ROAmZmZX40x07VabdcJsvyFSqeTsrKygvc9X1UlxoiqcvLk2xRF8fKVq1e+3Gg0Pp6fn/9uaWkpbAPEGI8uLCyUBs6jdvmnHxER2p02169fH4JVlWOzszjnOHVqwY6NjU1d/eXnD4zENvDDToB1zrG+vo73HhFBVRER0jRlfHycyQOTzM3NgW4ByuUym5ubqCrT0zPu9p3bb67dWzuyCxBCUBGhKIohYGAhBIw15HnOjZs3EInDU8y+cYxut4sxhqSUYI2txBjLu2oQY9QQPDFGfIjD4DEKURRjHGPjNQ4ffmWYOoBypQrGIgrWJoBFjbPzn11Kls68FTFGEwDvPSFE1luBa8uBol8iUUOzdZCqr7D+/Z9b6RkseNhbqRJC5O8HB0r1ytxCtnZv+viHX9+4hn4xBMQYqbeEDhMcPDhJpZwMooEBw2gH665uqwBHjr7uDoseW623Xrt5t34RzOcjKQo0M+HQS/s5/uohXOKIolvD8gQN3ltjsBbz2x/33a2/6sm2GsQYyT1MVkpkRaTd8nR9ZC9KnKVWdYQowRqb7wJkQalWSr2dGIOzlr3JEKIiirfWpEOA915D8GTeMF4t94MbEvfsgLyIOAsSFRUtjKGzo00jWQH7qgnWGpwBfYb4UZS8EHwQKlUHqsQoObA5CmBQA1WDD4qMNIoqiPae9Sa8F9hHQUS39VTuA7kPmahsjE6yhBDIvVBEJXYjaREovOCjEkTQHZ2pW0MxUmRDlhVspkUWoj4arQEhBIpoaKaBNARyH1Ht7V7/o/d3yhpotrvSTouOEWkMAYBmhZC4hFY3UgiEqOxVRVTSLMQ096lamtsAaSFMTlQZqzhqzm6rwTPPgTU8eiyS5SHTaFoDgANY3fB0vbC89hjrHP9HivLgYdNkaZ6Lzx8PAKUQwu93VluzDdmXdJbrbP139yZRJE87jazTWmms3KoDJgFMu93+ZnP1/j++ouWn3zxPnOQYi+x+0X50eXnpq+bgnkqAF4ED/XQ9D0GBbn/INgBv+gFdH/Q8wQcABWLf9F8A0n+tNyUc0gAAAABJRU5ErkJggg=='
saveBtnIconData = b'iVBORw0KGgoAAAANSUhEUgAAABYAAAAWCAYAAADEtGw7AAAABmJLR0QA/wD/AP+gvaeTAAAACXBIWXMAAA3XAAAN1wFCKJt4AAAAB3RJTUUH1QsKDTQVSGVyBgAABAtJREFUOMutlV1sFFUUx38z+zmdbbvTZZfd1mVZPpYiBdqSVosfWBMfMEI0RnzQF9P4ADVIVVID+iBGAhiLaWwfRB80ipLIiw9oDDE2xog0QVSsJbS1NsXW3bbTdr9nZ3d8aHdsqVBM/Cc35+aec/73f889M1dgAZ57o/39PxKVj8amtIqF65LNSPpc8b6V8m9tna9+9h23AaE46fqgrfbsj+GLqyMbs6GVPjGeE51VZU7xz7jGVDrL9NXzQ6lkxhFW+tu7Xzv98XLEliLp+f7A2VVbHnAPD/TbB4eHbSO/D4ojE5PYSitQnHac3nXKdLpQdj3m3LnrEc+li9/8OnArYhHgh2vO43essA4nBr662ui98HRPxz7xvlDvztzM9dnRwStGdDZOPJlA8QWQfUHHL7Gtn770+i55WWI1Ja2z5BMz968ffejoi+99AnDs4EdfNgRHjyRUVUjOTqFlUgD4gutFr6yNTGjh7mVr/NShFz4PO3I/z6qFwwudDkng26kQqzfVL0pyp/uY+GucysL0jXztnW93nQCwAgTL4l+kJu3dx46+uWTnB9u6lqxlPHcxOtTDhycX6eDlQwePAyfMUqTG7dcUt5dUau64yWTSHCZZ+p+5xWZfFDcyOkIymcTr9bL/QGutqRhY6/cHAEgkE7hk178qLcIoFAB47MhpdE3jzOE9ACiKQiwWiwCXxfnY3XVb60mn0wgIptLqhh1UN+ygXPECkM1m5mw6zrrau1lT0wBAiVQCwD1N9wJsNksBPFxZWbVI1bvPNzPw0/cAjI0OgWGQ13Nk0kniMyoAQ1d6eWffdjMnGAwBvAIg7j/QWitJJdjtdjwejxkU8IboaNlGf28PFd5KCoYxXweDMreHgcsX6GjZRqgyAoAsy/Pl8LD/QGutCERW+vwAqKqKLMvIsowoioSrNtC5t8kkFwUBV3kFA5cv0Lm3iXDVBjNeVedOUV5WDhARgc13btxELpczrFYLMzMzqKqKJEnIskxNpJ5Tbc309/bgXhFguO8Sp9qaqYnUm4SqqpLP6yQSCWPtmrUATWI2m93i9/txuVyCrucpzN94MUFVVSpKA3Q820B/bw9vPVNPRWnA9AHk8zoIoCiKEAyuIpPJllodDsfuSCRitsvNoCiNfH2ycdnfZXV1NU6no8UKEIvFmJycvHnfFi/uNnxe71xrFj8QNE3D4/Fw7tw5HA4HsquEUlcZoigiSRKCICBJEi6XC1VV8fl8RKPRJbZILC7cTdd1LBYLhmEgChay2SyapmEYBjab7ZbKb4SpWBAE0uk0dXW16Hoei8WCrutmoN1u57/AChCNRg2/3z//THmWTXK73QCEw+EldmxszAAE4YknH98TCPjP8D9ibGx8+9/nYJ9TcCLRagAAAABJRU5ErkJggg=='
saveAsBtnIconData = b'iVBORw0KGgoAAAANSUhEUgAAABYAAAAWCAYAAADEtGw7AAAABmJLR0QA/wD/AP+gvaeTAAAACXBIWXMAAA3XAAAN1wFCKJt4AAAAB3RJTUUH1QsKDTkjMnGZ0gAAA9ZJREFUOMutlF9MW1Ucxz/3UvqHC5TbrqUtVlZgHXNsQ5ZOmX8mJj7MuEVjnA8aY8LTxgLUbWI2fXDqHNMwJUKM6INGp0vciy8asweJMeJIJupEtkHFphOkjAv0fwu9PrRU6tzAhG9yc8495/v7nu/53nOuwDIcfK3jgz/CjkeDM0nT8nFDoRqxFoeGy6XfvN0vff4dq4Cw1On50Ft/7kfXhfXuTYnKcqsYSon6ilK9+GcoyUwswezl875oJK5zySMdvS+f+WQl4YIl0fMj9nO3b32gbHx0RDs2Pl7o/31M9E9fp7DEhKzXorfUyLOxdOm1oH73nkfMFy988+vorYRFgB+u6jtvW6cZD49+fXmHZeDp/q4D4n2Vg7tTc9fmA2OX1Kn5EKFIGNlqR7I6db8Et312+JU90orCStRQU7AYnrt/Q+ChE4fe/xTg5JGPv/I4A8fDiiJE5mdIxqMAWJ0bRIuU9E8nXb0rZvzU0ee+cOlSP88r6WPLJ3UGgW9nKlm/uSGvqCw2zPRfkzjSs//W6+h+q+cUgAbAWRr6Mnpd23vyxBs3rPygt+eGsbj5LgK+fj46neeDF44e6QRO5aKITmqvymUWotHMdiORSO7JicX+6RcUavN4/oCfSCSCxWKhtb2lPucYqLbZ7ACEI2GKpeL/dLoENZ0G4LHjZ1hIJjl7bB8AsiwTDAbdwJCY5e69c1sDsVgMASHntNazi1rPLoyyBYBEIp5pYyFq6u+mqs4DQJGhCIB7Gu8F2JKLAnjY4ajIc/VeWxOjP30PwETAB6rK4kKKeCxCaE4BwHdpkHcO7MzVOJ2VAC8CiK3tLfUGQxFarRaz2Zwj2S2VdDVvZ2SwH5PFQVpVszmolJaZGR0aoKt5O5UONwCSJGXjMNPa3lIvAu5yqw0ARVGQJAlJkhBFEVfFRrr3N+bERUGg2GhidGiA7v2NuCo25viKktmFsdQI4BaBLXds2kwqlVI1mgLm5uZQFAWDwYAkSdS5G+jzNjEy2E/ZOjvjwxfp8zZR527ICSqKwuLiAuFwWK2uqgZo1CQSia02m43XO18VbnWTPCVJBgdhR8kV+t69ctML9+wzzcTjiRKNTqfb63ZncmpvO4ygqqgIgAqCAKq61GRLhSwHqqqq8Pl8CELmEp9++01qa2vR63XNGoBgMJg9Ngb8fv9qfrfU1NQAYDLJ2cOVWXlJS1xONskmVgtZljMfy1iG0ViK0WjMm9csfynUFuLxeFgL5Am3eQ/+r+LW9pab1mgApqam1EPe54W1cDoxMaECgvDEk4/vs9ttZ1lDTExM7vwbD9hwGydusEoAAAAASUVORK5CYII='
reloadBtnIconData = b'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'
toolsBtnIconData = b'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'
addXnameBtnIconData = b'iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADdYAAA3WAZBveZwAAAQsSURBVEhL7VVpa1xVGH7OXWbNMksmk6VJJsk046QoFgStRVFaF6h+EvwBIuIPkILfRBCl+E0UxQUpCn6QWhCF4oo2UqQtkjSJk2iSmaa9zcyknTWz3HvP9T1nliYiCKkffYaX96zv825nLv5LOA6YA7D2VGLPZL/gxckZMPtuqBgikzYc5SpTXYvMm9q4YwJemkhCYydgO4+Q/1NEYNFyilz/AUz7kn10+v3TWSN7gjb3RXbs4aWvZmJLhzzexAGmJXxAwwFfvAX790uaYn/CTr35evWF51/0Mba/YDT22Y6LndWZdp/KtGcYnCK4eYbbjV8yuULvFxrn3NfX1w9B0Ly5ifOfvops+go8PUHc+/hzGDt8HKq3nwro7BG612KwQj6GABQUqcqXaKEORWmoDX6gP5ePJ9Tjjx175eiDD0mCnz88iYjfQHRYQ7gPuLG+DJv1QA+OYWtrC8ViEalUCpZlwe12Y25uDi4Xh8JKUJUtKE6KipImB3QY+Zj+9TktorXcaGH9t28w9dRhCtst52upRXiMP9Eb30Emk8H29jYajQbq9brcT6fTZHiAnAthapxhIGxCVfvA2QwsJ6JZ/NewIk+2wTQNnLyzLQ7HtsEdTuJIbyORCIaHhxGPx+XY5XJhdnYW4YFJ9PTeT8V4Ag3+NOrWkzDtByiKcWmzS7C2fhFldwCrK1exsZbB6moGWzsOjGIO5YqB0dFRTExMYGRkBKFQiPKsyHk4HMbQ0DRU7SBMK0nGp2Dz/rbVXQTnvnsbaV7DhZUV/HR5Hj9eXsBSoYiL6xcwf+VbNJtNWYNSqYRyudzVlUpF6mq1CtM0ZfFFE3TQJVhY+B7qQBbVaS/y0z7kJ9yohSswbs5jc3MJuq7L1HRkcHBwjwSDQXmm02UddAmOHHkWecNE+o8CjEwF+Rs1FPIN9PpjiEaTMoJCoYBsNis7arcWe8JzITbVbjcBe+PUa87Jl16mMEt0wEZ15xYdsOWm20UtqnupS5SuZx3ppOLvYyGiAYrFAs6c/fx2BNeuXacND8olE35fBPWaAsPI0cFWrg3DkIZEa4o3o6oqllMLlPcm1jdWpYPXjU3ktrOSpIMugVgURRKXhSFRtFKpLMMXD6tWq8kzYr0DzsljoYXn7Z+4K6SDLkEH0WhUFmt8fBzJZFIWNBAIyJbU6J0kEgnpvZBDyXvg9/txMH4XFVm0rkqetg210SUQj0j0tjAiLos8ejweSSbWxGMT2uv1dgl8Ph9cuotS6ofP68NkbAqx2KQk7YC9895b50PB8FEuqy9WWht3CuqmcqVaXmYffPzuo7lsfszhre8B5VsVQvXQmqap01izpLY1uiT/u8h7S9NIdN2kqCwXaYqU1jRbiDhD88ZAKJD7J38pkTJ1QguDQu8eCwgjQsTXa7cWcrvC/+PfAfwFfN8ubfJ/ddoAAAAASUVORK5CYII='
chgXnameBtnIconData = b'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'
aboutBtnIconData = b'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'
slotExportIconData = b'iVBORw0KGgoAAAANSUhEUgAAABYAAAAWCAYAAADEtGw7AAADjklEQVQ4y9WUW2xURRiAv3Ppbm+02miJlF6wNPRGwdZbi7UUKJfGaFtIY2LAJxJtkEQk+uClvmAUWUIIVEMwBjUhhoSAAmnaFSzY1lsUsKlZ0rRU2c3uJghnu7une87ZMz4sbLNYElN88U8mk5n555v/Nj/830S6V4C4vhiUGKSBlPXnfwMW0SJABluAJCVo0zJS3ngq+P3du74HnpjjOz+80XXoSWSBlPnHDPiD3e/9nJ8/v7btuY45eXHi5HERDAZ+ef3lQ4+SbaEm3ULUbdrYiaZpeL3elEv+v3wMjPbyW/AcEWscRZ5HlrqQdUs201DRREFBAZs2dkoHP9pfhwxElRkwgGmaBAIBVHVme3j0PF9eeoeiooU0tyylsGQtoZtTeH1eTg3vYXDiDNvWdFNTtSxpopQ1iZxSIpKEoijJ8fWPxzg28i7rn22kraOVvIdhRD/BhKMXZ1mQLVvbsTLG+fxCD5J0K4IigfwHWFVVVFUlFNXoG3ORe18O7tM/0XPgMy5+d5XKnLWARMj044n20dbeypUbvUz6JhKM7MSszgYGODJwABCk6cXsWLMTwzLYf3Y7zjQHRXWV+KZH0M0wobRrFC96iI9P76GEqiTrrhZPar+SoRSy64XDVCyqZllZLa8072Pk8jjzWEAsGsfQ4/hujlFaUsY1zZOS8LtaXJhbTX1pC+nO9OR5bfnj6N9cR53OwYja2LZNXNygND2HiHkpFbx334cHDcPouhP8SNEKHquoT6mQZGnaCoZuE4/HMTFw3p+JaU/d/mTC4XD0qIZhdG3f9iq2bSOESGa3vel5ZEnGc3WUt45vSAHLQiEWsbBtgRAGzgczsEWErwLdt1W6ZADDMNA0jVAoNHNZSoR/SUklO1q+QJYyaWxYyYudL2EagljUIhaNE4uYiLjMlvYu1jd2oMhZLM/enIhxLBZDlmWEELN+1/qljYSjLj4Z3omzIYsr2hCWMaP77dhRqvKacQ/2sfWpvfg8/gT40yOH/1U/KHE0MTDkpqWplYvBPgxLx6FmUP3ASvrPn6HYsQqfxz9723y7+82CcHhKUWRlATAfify4FZdNy1otAaNp7uW2w7+4ecVq6XLwLDX5qzg36LZjhj445LKevqd+3PCaOpKdkVvR0viM3H/hlB3Wtd+HXFZ1SoLn2HtrwrrmPdl/lLCueYGaOxXmBB5yWTZQbgu7Hyi/tU6RvwGCiW4Z2KCRQQAAAABJRU5ErkJggg=='
slotImportIconData = b'iVBORw0KGgoAAAANSUhEUgAAABYAAAAWCAYAAADEtGw7AAADcElEQVQ4y9WVf2jUZRzHX8/z/d7pbvNcm5u1o1hlkUEG6jh3Yo3KAom0NjYsIigGa0lliwKjJKRIZW2FUqxkZv5jwTB0C2lbuubW4VaeKepWcz+6xZ1b7W632+573/s+/XHs5JjJWP3TA58/Hp738/o874fn83ng/zbEvwWo8RWgxcAGInPkvwGr6G2ABEuBEEnajETkDKSDP9jz3o+Ae4F5vG9WN6xDKoRj+Bp49573e/Lzl6/esvmpBbk4+k2TCgYDP73xYsNaskz0lC3UmrLSckKhEH6/P92yUgAkLJNjZ77iiaIKNC21FZfLRVlpudj/ycdrkEBUuwYGiMfjBAIBdF2fe59Ksa9lF4Ph7/g99BuvbnoXIZLmAoEADocjdUSROYRMeyJCoGnadeNAWz1jcR+bN25lLO7jQFt92vpsElQSOQes6/qcONLRSH+oFY+7CK//KB53Ef2hVo78cDClmQWLrCvzAzd7m+j2H6LEs4FzwXYMc5pzwXZKPBvoHjlIs7cpDTw7bgju8LXSfKmWh9dv5PzYSQxzGgDDnOb82EkeKH6Q45dq6fC1zh/ce9nLlz07eGT9Y+Tl3EzJiq0ICVIX6HbBQyufxlVwK551RRw6s4PTP59KL+kP6/fuNwyjGmBb9StMTEwAUF5XmCZ8rqyKtsFGhFBIKXj0rko+O7wvTVO8tIq8xbdgs+mf6oZhVL+8bTuWZaGUSln6+rWh1IaKupXEzGkyMu1YxNE0DUsk0nROpxMhBFf/CrKzYXuVDmAYBlNTU/9YVbrIIhaPkpFhxxQWUkqETIItKzkPh8MAdJ3t4Gp8IHnHsVgsKRbiuuGw5RGZCZPtvIlFDg27Q2IuDuOwL+NsX29KZymLxs5dRBKjycpr/OLzG/YBK6H4dfAy7jvv4Up0HKkJJhnl3lWF7G55lrWdz6AJHV+khWhimGx91dy2+fbOt1yRyKSmSa0AWI4gfzQxcP+Q+Palysrn6YufIGZNIZDct3QTXad6rYsX+kApnNlL+HN8QoJaPe9O5nld71+Wm3NHacWTcjDWzWQ8gCZs3J7pJtdeSN/ARU4cb1cJZbac3ms+Pn9wjX43gk67vijX4ymWrgIXzuwljAz+wS8XfNbw8JBQqLquWrNmQT+Ip0b/CMELQAYKCcwAPcA7XbXm97O6vwHpzFBss4CGCwAAAABJRU5ErkJggg=='
slotClearIconData = b'iVBORw0KGgoAAAANSUhEUgAAABYAAAAWCAYAAADEtGw7AAADkklEQVQ4y9WVbWhbZRTHf/e593Zb09Y225LOjHXtqrPIHFvtVOhGnTJQaBVXhuIHYVSwdSh+qH4QX0Aqs2ObimUwhlW3L4JMnS9oVVSk1Na1Ol3s2q4LfQkuSe2aNLnNzc29jx9C08WuDKtfPHA+PYffOf8/nPPA/y2UfwuQf1aCaoIOimvivwFLYwMgwJGgKBlaUqC4L+WCD7a3/Qjcscw+vc+1HL8TIVHyxxfAr7W/etbj8W5/8IGHlqXio49Py3A4NPBs8/HbKUijZWUhqxv37iMajRIMBolOhTn3WSdp02Br/eO4S30L/im5vX0+H4179ykdx96sRgCGiri6wLIsQqEQmqZx9vRb+OQXVOb/QN+pF5mdDqPrOrquo2laToZCISzLyo6ouMZywYqioKoqqqpimwkKVgk8xTpV3gDnPj+JKgSRyT6Q6WzdfGZVyAxyEXh+im0NTzAUqcSyJaXuPJzZAIO9HUwONyOltWjqebBSEABY8PhqMMCmLdUY0f30fvs2W9aP4K3qx+uBhGsnOFa2binflwQDbL/7fgrdRQRHmtlWU8eG8h30vNHFxa6j6Pd4WLt+Nx7fbWj6ykXgJa3QNI0rkVHCk4eprdtDWdlmRo9+jfnBRbY2rmFnbREu8T793zxKYPBTUmYiV8GR1w91pFKpFoADLU8zMzOTfQwF/VzoP0ypO4nzXhh7OsGud+5lhVuAsgrUEqy0C7+/m9/9E8TNCgy7AiXPc0w52N4mnzrwDI7jIKUkFovldHYcm5/aXobAd9Sf3I8Qf4C6DuwQ2EGcdIykU04y5WZy8jwTE3OELk9mPE6lUiQSiWtulBAqGx97mHhkHKHngRUHOQpOFOwwSWOaeOI8CSPJ2pK7WL36Jr76MpgBm6aJEAIp5TXhJd6NDP8yheMoKNYoUlpYqSRJM86cEcUwZpBSIrRCJsb8WFRlwJ3vnrjuLVijrmR8/DeKXfmkjF9JWyamaeA4aQDyS+7DmBMMDgaI2bsXn80XXnreF4/PqqpQbwS8KHjstC2KVlx+5Nby3tpddfVqPHIKx57NWKW5cZXs4UrUkn09H86m01ZDUyvf/6NL1nmE7ps376ip2FSlJ2M95OXfgtDLuDTSnR4aGog4DnVNrQwvWpDrhW3TMHSh7+fConW+G4prxNiY3xoZ6Uo7dvITx+HJplamlv2DnDhEmaIwoChI4Izj8EpTK4G/1/0F+xB8831SyvgAAAAASUVORK5CYII='
moveSlotUpIconData = b'iVBORw0KGgoAAAANSUhEUgAAABYAAAAWCAYAAADEtGw7AAAABHNCSVQICAgIfAhkiAAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAL/SURBVDiNtZRNaFxVFMd/5368eZkxRmQsbTEGxbZC0IWMpU5SbCmxILapFCO2LgSlWbhLbFJBcKErP0BcCIqCdeEiUAviQoJIwIgW3biwCoYGjTImTIXWTjsz7+O4yLzwUkuSgjlwuPdd7vmd/z33vCuqymaY2RTqZoLdRjfunXTvg2zfNh8PT01p8r+AByf9S9vv6DuGQWr627vA6Hox65ZiYMIdva1Ufvlg9WhpaM+RYs+t5eN7J4IX1ouTtbpicCyoFEulmacOniidr8+SpBG7ynv4dPrjxpVW48i3b0Rf3rTiR8al13o3/cSBZ0uLV+f46/IFapfmWLh8nsf2PVnyzp0ZnAx33RR4cFK6U2dnDu97pie1LS5c/BFnHNZ4fl36nkZSZ6h6qBuNv6qOye0bAo+MiFXc54/uHundUt5qfln6Dmt8xx2I8MPCF5RuKcrAg0NbjHfTlVHx64Jrd9v3qv1Du/t3VPxPta+xYvHOUwgCfGBQ28Z45dzCWfruvMfdf+9D/V099vSa4IEX/fh9d1WO7a8Mhz/XZjFOCMMCYVggCD2RaeAKQlAw4CPO/XmGygMPh33bdg5XJ/ypPGulK6rj7vG+rTumnjt0qss5j5KikjAzf5pYWtSvzXMt+Wc5SuHwzpNoqmgKzXaLs9OfXK1f/OPp2bfizyD3gxgr478vznW98sHzAFgvzVdPfBj6wNFoLxLZBt4bBFAF44R3PnqtGbc17CCKOBkDVoNnX4/2X1cWNdbiA8+V1hK+YDAGEEFVsVaI2xp+82Yk19d3FfhGZozQiP9GbYwLBOMEEUFTRewNeRsEi6WZXkIMGCs4bzBOiNspsjZ3dVfIsjkRKWRrzbgBCppCmihJpGgC6EpMUUQKIrJKpMuAgO98u84cVTjQO4pKirEgIizfHqTpyhvTDcRAJCJxNne5BDYHdUD95NvHy2sfmHomImcKJKKqmeI8NBuzZKbjHb2kHY+BJFMJtIG2qqb/eTY7SWzOM6jtbElz8CQHV83B/gXtSQriGSyg6AAAAABJRU5ErkJggg=='
moveSlotDownIconData = b'iVBORw0KGgoAAAANSUhEUgAAABYAAAAWCAYAAADEtGw7AAAABHNCSVQICAgIfAhkiAAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAAAL8SURBVDiNtZRPbBRVHMc/781Md1bTGMCkhYNtOIiAnkw0bSFpNCGeOEAwMRyMMaReNCY0JXpFozQWT3JWExsEIzaxJkSREGiJIglRNCqVDSEIxlKqsN3ZeX9+HnZ2O112Wznwkl/e78283+f33nd+v1Eiwv0Y+r5QgbDdi4GR6AaermWjNX9NjZruewLj6Xp76COsNzhvsT7FeYvzBusNqUs4dPittonbgwFQnP7jCFW7QNUukLqk4T+3cc8Kl7nHoVD/a98KYEFYrBqR2lrEs1I1LQtehHgEX3signHVFU/c0HjLSHRSPIONjKFKRHws4vF4vNQsdRUSW27sGRiOFo+uOTU1agaXgL2TsZ7uR5/es/3NYhhEOLGx8xYvFrRHh4L3CQm3iAoaHcAru96IRcCYKhMnxhdmb/15sM5Tea0GhqO9m9Y/uX/H4MvFC9e+oWLnsVSp+jvMVa+R2H9BAQIiEOtOtvbs5uTUVwuXr8/snx4177bUeOo9M/brlR/Gvz13LNm8rp8gVBhV5u/qDFbfIYo1HbEmijXFYszW9bs4f/FscuXGpYk89C4wwNqSG5r+5evvf7z0ndm4tp/ZtITq8EQZMIoDOuKAvp4dlK5etj/9fu7nyrx7sZmjWpXNln2qU6nowvZnX+gNH6roi/PHCUKNDhVKweOrtlGZ0zIxefS6M/aJ6YMy18xoWW5nDshtbczg5Imj/xTcKjas6SfsUEQFzYaH+yi61Xx5/IvbQvBMK2hbMMCpMblqU7vt2OSn5e5oE+sefKxmhc0c/ny8XE3NzjMHkt/axbeUIj8GRsKdqzvXfLz7+ZceCCLFJ599WL45e3Pf6dH0g+XiloCVUgoIcqYB3fdasPeR3t7hoKBUaaZ05Oz77nXA58wBTkTsEnAGjKg1TJjzg2zWT70avBMEuuv8ITOUppgMaDOoBUw2W8DUwVEO2Ow3r00OYtqs03pL16/Q/E+sX1NnlvVdQ4LmE6dAKiL+ro/XTufMryerw10OLpKD/QceAIZcIO5IFQAAAABJRU5ErkJggg=='
exportPresetIconData = b'iVBORw0KGgoAAAANSUhEUgAAABYAAAAWCAYAAADEtGw7AAADf0lEQVQ4y7WUf2iUdRzHX8/3nu2msmudczjxuGM2MleL6KyWONtmLgu3nURIvyiHxiQoMBDUsEKi/hGCAuGe2qo/CqScINfcplN0O+iPEZvr0pRd4uy6Dbe7tvv5PM+3P7a77XaKBO4DHz5fvt/n+/68n8/n830rUkqWwgRLZEsGrHZ0dBzXdX3PfUxiqqrqxev1GvI+m9frNdQM02AwiKIo2bSZ9eK4eJ1pfiY6HA4AoS78eGJigkQikQcSngxxIdDF5fE+ZvRRLMJGiXUdr2xso97dmAOcrXG2i0JgsVgQQuQA/3rlEieGP8bpdFD3fDUOVyPRqX8ZuzXGt/5DdA2d4MNdx7CteCAHXCwELisrw+l04nQ6cblcDPzZx8nAp2xvrqVl54vYK+ByvJPRwi6slWHe2NNErCDAl6c+QwiBECKfsaIo+Hy+7MF0MspPo0dYubKU7tP9nLUM8vCGtTxTu43hiI9oOsQVo5tmzwt4vd9wI7QPZ7krn3Hm91tbd9PaupsbegCQ2MxqGla/x+Ed3xMY+pvBgQDlRRsAiKeniXITV4UD7cyxHMY5pZhNIIhEphj55xLLVRcdB05TXuzgiUfcHGzWGBkOUswakjGDVNzg1tQ1HnKtZyxyNac36p1GqKTETmWpm4bHm1BVNZt4Y9XTxE6OoyZspGImpmliyEnWFdmIpPrvXmMATdMAMOIKwT/+QhvVssCZi9K0kIqbGIZBmhTWB5cjZeLujD0eD+8fPjK3U8zgzSAxdZyQ9WfaP2qbr5+0kJzRMU2JlCmsq5Yh0Xnp6Ko7AyuKgrWoKGfQrTjY29LJ0VOvs6nGTcXa9aRTkmRMxzRBmhJpCN707CM8EaK3v4ea0rfygdWCwjxV2eJuYDp+nC962rA+u4KrkQH01PxjOH/tB6rsdfT297B/u0b4+u154Az454cOZF9fJiqKwo4tHqIzU3znP8jW2gZ+C3eT0uMUqst4tPQ5ei+e5bWnPqGpbifadQ0VMAFht9vvqYdvt7xDZGYSn/8rNtfUMxQ+R3VZPRf8fpoe28/el9/NSqfS3t7+v/W4O/gjUWWErZsb6b14BpusYptrV1bwhBBfKxnhUGZnRZ3zAqBwgVvn9hRAt67GePJVtU8IscaU5ljgF6Nq8neZBNJSSiNnKuRshvScx+/FetMHBZWmNDuBltsjZmzx+X9EYa7x+4yIYAAAAABJRU5ErkJggg=='
importPresetIconData = b'iVBORw0KGgoAAAANSUhEUgAAABYAAAAWCAYAAADEtGw7AAADV0lEQVQ4y7WVXWgcVRTHf/fuzGyW7LabLcaYGIi2AVNQwYeG3cRQYyJalJJoaOuL0iDaNy0VEUNiUASLxfqBTdnFhLYSKSKCzSKhNkmTWNRiU220WGu2iTHpstJsU7tfszM+pDNmuxtQaA4czh048zv/OefeO8I0TVbDJKtkqwZW+vr6enRdf+4WFjEURQkSDAaz5i22YDCYVSylkUgEIYRd1lrfHG9eW8O3YmVlJYBUlifHYjGSyeSKkKyR5djpo2xreJaysjsKgvOGJ6XE4XAgpSzoQgoOHH+L8T8Osn+gGyEEQojcnGUilOXg0tJSW6Xl1vPe/i6uZCfZ2ryDE2ND7Dvazcvbu221Qogc1TmtCIfDBcd8cmaAef07HgzU8u3sF/hrGxka76ftjRHcsxsB6Dt4YGUwQHv7zhzooS9DTF/4muaGR5mIDpLWE/wYPcHDdY8wODJA+s61lMRrcr4wr8dLBSTx+ALx+AKfhg/z8dirNNU/xrnYMGk9AUBaT3AuNszmQDMJ5wQJ33xej2Wh6Xu9PiYunKFn5CWa6pq4zVfG5g07EBKkIlA0QWPN01SUV1Lvr+eKY5i9oTdtcQVbEQqFAOg9vwuAr0Y/B+CZp17AWawihImUAlUTBI/02KDR6AdsnWrFX1KPaZq54JaWFl7s6ALgbp63X4q4j5DSE7iKNQwyOBwODJEFYKDjMl6vj1AoxMb19zIfm+OH6GguWAiBs6gob1eocg2pzHVcLg1dGDf29RLY7fFgmoY9p1Nnx5hNnM8HK6qWB3art3MteRXvuhIWs1GEFOhFV3GqHl5/t4Mqb7V9MnuOd5LKLvwLtuBvv/aKPWEr7vlohouXvmfT+mqmrv+FdAgW+ZP77q/hmzP9bGn8BE3V2PV+G9cyv1HqrEUBDED6fL4V78HdbV3s/HATTY0P4VInSRl/M5f8mQcCW0jrGTo/2w4meLxOQNBQ3mqI3t7e/3Qfh6cOYXgu0rrtCSKpUyxmLuMQKncV17JOq+LX339h8NhJqt2P469oDgnrGIql/abccBXQlrnzniflBl+VPKypzrWBgF9WlFewxuthJjLHT5NnjenpS8I0zf3j+zK7AcT//ZnW7VHfQ9AOuDCRQBI4DXSOv5MZsvL+AWGthETiHqtEAAAAAElFTkSuQmCC'


## defines main window, frames and their weights
root = tk.Tk()
root.title("BusyKip")
root.minsize (425,300)
smallIcon = tk.PhotoImage(data=sIconData)
largeIcon = tk.PhotoImage(data=lIconData)
root.iconphoto(False, largeIcon, smallIcon)

mainFrame = ttk.Frame(root, padding=(5,10,5,10))
mainFrame.grid(row = 0, column = 0, sticky="NSEW")
topFrame = ttk.Frame(mainFrame, padding=(5,0,5,10))
topFrame.grid(row = 0, column = 0, sticky="NSEW")
midFrame = ttk.Frame(mainFrame, padding=(5,10,5,5))
midFrame.grid(row = 1, column = 0, sticky='NSEW')
midFrameLeft = ttk.Frame(midFrame, padding=(5,0,5,0))
midFrameLeft.grid(row = 0, column = 0, sticky="NSEW")
midFrameRight = ttk.Frame(midFrame, padding=(5,0,5,0))
midFrameRight.grid(row = 0, column = 1, sticky="NSEW")
bottomFrame = ttk.Frame(mainFrame, padding=(0,5,0,0))
bottomFrame.grid(row = 2, column = 0, sticky="NSEW")

root.columnconfigure(0, weight=1)
root.rowconfigure(0, weight=1)
mainFrame.columnconfigure(0, weight=1)
mainFrame.rowconfigure(1, weight=1)
topFrame.columnconfigure(0, weight=0)
topFrame.columnconfigure(1, weight=0)
topFrame.columnconfigure(2, weight=0, minsize=100)
topFrame.columnconfigure(3, weight=0, minsize=50)
topFrame.columnconfigure(4, weight=1)
topFrame.rowconfigure(0, weight=0)
midFrame.columnconfigure(0, weight=6)
midFrame.columnconfigure(1, weight=1)
midFrame.rowconfigure(0, weight=1)
midFrameLeft.columnconfigure(0, weight=1)
midFrameLeft.columnconfigure(1, weight=0)
midFrameLeft.rowconfigure(0, weight=1)
midFrameLeft.rowconfigure(1, weight=0)
midFrameRight.columnconfigure(0, weight=0)
midFrameRight.columnconfigure(1, weight=1)
bottomFrame.columnconfigure(0, weight=1)

## loads button icon data for use
openBtnIcon = tk.PhotoImage(data=openBtnIconData)
saveBtnIcon = tk.PhotoImage(data=saveBtnIconData)
saveAsBtnIcon = tk.PhotoImage(data=saveAsBtnIconData)
reloadBtnIcon = tk.PhotoImage(data=reloadBtnIconData)
toolsBtnIcon = tk.PhotoImage(data=toolsBtnIconData)
chgXnameBtnIcon = tk.PhotoImage(data=chgXnameBtnIconData)
aboutBtnIcon = tk.PhotoImage(data=aboutBtnIconData)
slotImportIcon = tk.PhotoImage(data=slotImportIconData)

## defines tab button style
tabButton = ttk.Style()
tabButton.configure('TabButton.TButton', anchor='w')

## global variables
openFolderPath = ''
openFolderPathPrev = ''
folderPathStr = tk.StringVar()
statusStr = tk.StringVar()
statusStr.set("Ready")
reloadFlag = False

presetSlotList =[]
presetSlotListFe=[]
presetSlotsVar = tk.StringVar()
selectedSlot = 0
renameList = []
nameText = tk.StringVar()
presetCount = 0

vanillaPresets = {
    "fae6f5587feb39f55be72ad7db315acd":"M3GTRCAREERSTART",
    "59c1c4e75d6edd1eb41b83ab7d9bf7a6":"BL15",
    "7b4657c1c099f5a6e2d69467f3564b62":"BL14",
    "e9169ffd911f8e9847f4bbe898c6c2c2":"BL13",
    "b5d77ff166efcf0057981d239f1c1833":"BL12",
    "e47407a2212b44f90ac0fc0cb3f2e7b6":"BL11",
    "1c37be570e1986efc8b9498bcd66a73e":"BL10",
    "aa5f05ad789c7f08c8b72bead02d619d":"BL9",
    "1c28e1236330521783e28be9ec48d071":"BL8",
    "1de607bfc7a8b85b3404513bb35346f8":"BL7",
    "fb41d097ccead87d6c86bc8322d9b9de":"BL6",
    "f2e43779158f8e82c9fdc5b06ada384c":"BL5",
    "97f6ce979bdd5c34e08595152280991f":"BL4",
    "fc4aa1184ee15eea1e8846db5f824830":"BL3",
    "8e3238a27b7a190f89975d600f406937":"BL2",
    "6424d6ded7253c496eb2777beef2f40a":"E3_DEMO_BMW",
    "fe4b3a0b35613990baaa6b8daf40a121":"BONUS_SL65",
    "73a309d5268a57d7a8be32281215cddd":"BONUS_C6R",
    "856358fc5b4abe7a631d4a1cb769557f":"BONUS_GT2",
    "d32b93fe0a557bd80cf19852c869c344":"CASTROLGT",
    "b6984ae3e6b750cd1459911a403e83f4":"CE_CORVETTE",
    "2eb7422a3142f44784e1776a356fecda":"CE_997S",
    "27dc5d77a8c5a6f361a5f02ba6c3faa1":"CE_ELISE",
    "50329777bc76bfac6634a8a618062d27":"CE_SL500",
    "1cef7bef462d870c9bfbe71073dcc6cb":"CE_SUPRA",
    "d9e7007a5277a5b1bd87a4af2ab7d678":"CE_C6R",
    "e15abbcd0f3871c68e0e079453f42758":"CE_SL65",
    "ce44b3c29c8f1bdd6c9e605ca49ce043":"CE_GT2",
    "78742301dc53c4b1ff17baa4b3b4ed1e":"CE_CAMARO",
    "8a7049ba86e682a32d258dff38d2138b":"CE_GTRSTREET",
    }

userPresets = {}

userDirPaths = {
            "openProfileDir":"",
            "importSlotDir":"",
            "exportSlotDir":"",
            }
    
## functions to validate characters inputted in text boxes
def inputCallback(string, newString):
    spChars = ['', '_']
    if len(newString) > 32:
            return False
    if len(string) > 32:
            return False

    for i in str(string):
        if i.isalnum():
            return True
        elif i in spChars:
            return True
        else:
            return False

## loads list of user-defined slot preset names from a text file
def loadSlotPresetNames():
    global userPresets
    
    if os.path.isfile("BusyKip_userPresets.txt") == True:
        with open ("BusyKip_userPresets.txt", 'r') as slotPresetNamesFile:
            userPresets = json.load(slotPresetNamesFile, parse_int=True)
            slotPresetNamesFile.close()
    else:
        pass

## function to load list of recent folders for profiles, presets and slots from a file
def loadUserDirPaths():
    global userDirPaths
    
    if os.path.isfile("BusyKip_userDirPaths.txt") == True:
        with open ("BusyKip_userDirPaths.txt", 'r') as userDirPathsFile:
            userDirPaths = json.load(userDirPathsFile, parse_int=True)
            userDirPathsFile.close()
    else:
        pass

## writes list of recent folders for profiles, presets and slots to a file
def saveUserDirPaths():
    global userDirPaths
    
    with open ("BusyKip_userDirPaths.txt", 'w') as userDirPathsFile:
        userDirPathsFile.write(json.dumps(userDirPaths, indent=4))
        userDirPathsFile.close

loadUserDirPaths()

## generates MD5 hash of imported preset/slot customization data and checks against slotPresetNames; if hash does not exist, it will be added alongside the preset/slot file name to it.
def slotPresetNameHash(presetData, presetName, checkOnly):
    global userPresets
    presetDataHash = hashlib.md5(presetData).hexdigest()

    if presetDataHash not in vanillaPresets.keys() and checkOnly == False:
        userPresets[presetDataHash] = presetName
        saveSlotPresetNames()
        
    return presetDataHash

## writes MD5 hash of preset to a file
def saveSlotPresetNames():
    global userPresets
    
    with open ("BusyKip_userPresets.txt", 'w') as slotPresetNamesFile:
        slotPresetNamesFile.write(json.dumps(userPresets, indent=4))
        slotPresetNamesFile.close

    loadSlotPresetNames()

## checks for case-senstive extensions in preset file paths
def checkExtCasePath(path):
    if os.path.exists(path + "/Presets/" + presetSlotList[selectedSlot][0] + ".bin"):
        caseExtPath = path + "/Presets/" + presetSlotList[selectedSlot][0] + ".bin"
        return caseExtPath
    else:
        caseExtPath = path + "/Presets/" + presetSlotList[selectedSlot][0] + ".BIN"
        return caseExtPath        

## clears presets slots when called    
def clearPresetSlots():
    global presetSlotList
    presetSlotList.clear()
    for i in range(presetCount):
        if i < 10:
            presetSlotList.append(list())
            presetSlotList[i].append(f'0{i}')
            presetSlotList[i].append('')
        else:
            presetSlotList.append(list())
            presetSlotList[i].append(f'{i}')
            presetSlotList[i].append('')

## populates preset list when called    
def presetSlotsPopulate():
    global presetSlotList
    global presetCount
    presetsList=[]
    presetFilelist = sorted(os.listdir(openFolderPath+"/Presets"))
    for file in presetFilelist:
        if file.endswith(".bin") or file.endswith(".BIN"):
            presetsList.append(file)
    presetCount = int(os.path.splitext(presetsList[-1])[0])+1
    clearPresetSlots()
    x=0
    while x < presetCount:
        if presetsList[x]=="empty":
            x+=1
        if x < 10:
            if f'0{x}.bin' in presetsList or f'0{x}.BIN' in presetsList:
                with open (openFolderPath + "/Presets/" + presetsList[x], 'rb') as presetData:
                    presetSlotList[x][1] = presetData.read()
                    x+=1
            else:
                presetsList.insert(x, "empty")
                x+=1
        elif x >= 10:
            if f'{x}.bin' in presetsList or f'{x}.BIN' in presetsList:
                with open (openFolderPath + "/Presets/" + presetsList[x], 'rb') as presetData:
                    presetSlotList[x][1] = presetData.read()
                    x+=1
            else:
                presetsList.insert(x, "empty")
                x+=1
        
    presetSlotListFe.clear()
    for i in range(presetCount):
        if presetSlotList[i][1] != '':
            presetSlotAlias = hashlib.md5(presetSlotList[i][1][72:656]).hexdigest()
            presetSlotXName = presetSlotList[i][1][8:39].rstrip(b'\x00').decode('ascii')
            if presetSlotAlias in vanillaPresets.keys():
                presetSlotListFe.append(f'{presetSlotList[i][0]} - {presetSlotXName} ({vanillaPresets[presetSlotAlias]})')
            elif presetSlotAlias in userPresets.keys():
                presetSlotListFe.append(f'{presetSlotList[i][0]} - {presetSlotXName} ({userPresets[presetSlotAlias]})')
            else:
                presetSlotListFe.append(f'{presetSlotList[i][0]} - {presetSlotXName}')
        else:
            presetSlotListFe.append(f'{presetSlotList[i][0]} - (empty)')
    presetSlotsVar.set(presetSlotListFe)

## opens profile's Presets folder and calls clear/load preset slots functions if valid Presets folder is found
def openFilesFolder(*args):
    global openFolderPath
    global openFolderPathPrev
    if openFolderPath != "":
        openFolderPathPrev = openFolderPath
    if reloadFlag == False:
        openFolderPath = filedialog.askdirectory(title="Choose profile folder")
    if openFolderPath == "":
        openFolderPath = openFolderPathPrev
        return
    if os.path.exists(openFolderPath + "/Presets") == False:
        badFolder = messagebox.showerror(title="Error", message='"Presets" folder not found in selected folder, please select another folder.')
        openFolderPath = openFolderPathPrev
        return
    userDirPaths["openProfileDir"] = openFolderPath
    saveUserDirPaths()
    openFilesFolderLabel()
    presetListbox['state'] = tk.NORMAL
    reloadFilesFolderBtn.state(['!disabled'])
    clearPresetSlots()
    loadSlotPresetNames()
    presetSlotsPopulate()
    

## sets open folder path label
def openFilesFolderLabel():
    folderLabel['textvariable'] = folderPathStr
    if len(f'{openFolderPath}') < 32:
        folderPathStr.set(f'{openFolderPath}')
    else:
        longpath1= os.path.splitdrive(openFolderPath)
        longpath2= os.path.split(longpath1[1])
        longpath3= os.path.split(longpath2[0])
        longpath4= os.path.split(longpath3[0])
        folderPathStr.set(f'{longpath1[0]}\\...\\{longpath4[1]}\\{longpath3[1]}\\{longpath2[1]}')
    
## reloads presets
def reloadFilesFolder(*args):
    global reloadFlag
    reloadFlag = True
    openFilesFolder()
    reloadFlag = False

## resets status bar label
def resetStatusLabel():
    global statusStr
    statusStr.set("Ready")

## loads selected slot when called    
def loadSlot(*args):
    global selectedSlot
    selectedSlot = presetListbox.curselection()[0]
    if presetSlotList[selectedSlot][1] == "":
        nameLbl.state(['disabled'])
        nameEntry.state(['disabled'])
        nameSaveBtn.state(['disabled'])
        nameSaveAsBtn.state(['disabled'])
        nameImportPresetBtn.state(['disabled'])
    else:
        nameText.set((presetSlotList[selectedSlot][1][40:71]).rstrip(b'\x00').decode('ascii'))
        nameLbl.state(['!disabled'])
        nameEntry.state(['!disabled'])
        nameSaveBtn.state(['!disabled'])
        nameSaveAsBtn.state(['!disabled'])
        nameImportPresetBtn.state(['!disabled'])
    if presetSlotList[selectedSlot][1] == "":
        nameText.set('')
    elif slotPresetNameHash(presetSlotList[selectedSlot][1][72:656], "", True) in vanillaPresets.keys():
        nameText.set(vanillaPresets[slotPresetNameHash(presetSlotList[selectedSlot][1][72:656], "", True)])
    elif slotPresetNameHash(presetSlotList[selectedSlot][1][72:656], "", True) in userPresets.keys():
        nameText.set(userPresets[slotPresetNameHash(presetSlotList[selectedSlot][1][72:656], "", True)])
    else:
        nameText.set((presetSlotList[selectedSlot][1][40:71]).rstrip(b'\x00').decode('ascii'))
    
## imports preset file to selected slot when called
def importSlot(*args):
    backupDate = str(datetime.datetime.now()).replace(":","-")
    selectedSlotFile = openFolderPath + "/Presets/" + presetSlotList[selectedSlot][0] + ".bin"
    selectedSlotBackup = openFolderPath + "/Presets/" + presetSlotList[selectedSlot][0] + f'_{backupDate}' + ".bak"
    if presetSlotList[selectedSlot][1] == "":
        return
    presetOpen = filedialog.askopenfilename(title="Import preset", filetypes=[("NFSMW Binary Preset", "*.bin *.BIN")])
    if presetOpen == "":
        return
    if os.path.exists(selectedSlotFile) == True:
        with open (presetOpen, 'rb') as importedPreset:
            importedPreset.seek(8)
            importedPresetXname = importedPreset.read(32)
            importedPreset.close()
        with open (selectedSlotFile, 'rb') as selectedPreset:
            selectedPreset.seek(8)
            selectedPresetXname = selectedPreset.read(32)
            selectedPreset.close()
        if importedPresetXname == selectedPresetXname:
            os.rename(selectedSlotFile, selectedSlotBackup)
            shutil.copy(presetOpen,selectedSlotFile)
        else:
            badFile = messagebox.showerror(title="Error", message="Preset has different XNAME from slot, please select a preset with the same XNAME as the slot.")
            return
    else:
        shutil.copy(presetOpen,selectedSlotFile)
    with open (presetOpen, "rb") as presetData:
        presetData.seek(40)
        presetName = presetData.read(32).rstrip(b'\x00').decode('ascii')
        presetData.seek(0)
        slotPresetNameHash(presetData.read()[72:656], presetName, False)
        presetData.close

    longpath1= os.path.splitdrive(selectedSlotFile)
    longpath2= os.path.split(longpath1[1])
    longpath3= os.path.split(longpath2[0])
    longpath4= os.path.split(longpath3[0])
        
    if len(f'{presetOpen}') < 72:
        statusStr.set(f'{presetOpen} imported to slot {selectedSlot}')
    else:
        statusStr.set(f'{longpath1[0]}\\...\\{longpath4[1]}\\{longpath3[1]}\\{longpath2[1]} imported to slot {selectedSlot}')
    statusLabel.after(5000, resetStatusLabel)
    presetSlotsPopulate()
    loadSlot()
    userDirPaths["importSlotDir"] = os.path.split(presetOpen)[0]
    saveUserDirPaths()

## saves preset slot file when called
def saveSlot(*args):
    if presetSlotList[selectedSlot][0] != "empty":
        with open (checkExtCasePath(openFolderPath), "r+b") as presetData:
            presetData.seek(40)
            presetData.write(b'\x00'*32)
            presetData.seek(40)
            presetData.write(nameText.get().encode('ascii'))
            slotPresetNameHash(presetSlotList[selectedSlot][1][72:656], nameText.get(), False)
            presetData.close()
            presetSlotsPopulate()
            loadSlot()
            statusStr.set(f'Slot {selectedSlot} saved')
            statusLabel.after(5000, resetStatusLabel)
    else:
        pass

## saves preset slot to another file when called
def saveSlotAs(*args):
    if presetSlotList[selectedSlot][0] != "empty":
        with open (checkExtCasePath(openFolderPath), "r+b") as presetData:
            savePresetPath = filedialog.asksaveasfilename(title="Save NFSMW preset as...", initialfile=nameEntry.get(), filetypes=[("NFSMW Binary preset", "*.bin")], defaultextension=['.bin'])
            if savePresetPath == "":
                return
            openPreset = presetData.read()
            presetData.seek(0)
            with open (savePresetPath, 'wb') as presetWrite:
                presetWrite.write(openPreset)
                presetWrite.seek(40)
                presetWrite.write(b'\x00'*32)
                presetWrite.seek(40)
                presetWrite.write(nameText.get().encode('ascii'))
                slotPresetNameHash(presetSlotList[selectedSlot][1][72:656], nameText.get(), False)
                presetWrite.close()
                presetSlotsPopulate()
                loadSlot()
                
            longpath1= os.path.splitdrive(savePresetPath)
            longpath2= os.path.split(longpath1[1])
            longpath3= os.path.split(longpath2[0])
            longpath4= os.path.split(longpath3[0])
                
            if len(f'{savePresetPath}') < 72:
                statusStr.set(f'Slot {selectedSlot} saved to {savePresetPath}')
            else:
                statusStr.set(f'Slot {selectedSlot} saved to {longpath1[0]}\\...\\{longpath4[1]}\\{longpath3[1]}\\{longpath2[1]}')
            statusLabel.after(5000, resetStatusLabel)

            userDirPaths["exportSlotDir"] = os.path.split(savePresetPath)[0]
            saveUserDirPaths()
    else:
        pass
    
## moves slot up when called, unused            
##def moveFileUp():
##    if selectedSlot >= 9 <= presetCount-1:
##        slotMoveOld = f'{openFolderPath}/Presets/{selectedSlot}.bin'
##        slotMoveNew = f'{openFolderPath}/Presets/{selectedSlot-1}.bin'
##    elif selectedSlot >= 1 <= 8:
##        slotMoveOld = f'{openFolderPath}/Presets/0{selectedSlot}.bin'
##        slotMoveNew = f'{openFolderPath}/Presets/0{selectedSlot-1}.bin'
##    elif selectedSlot == 0:
##        return
##    if os.path.exists(slotMoveNew) == True:
##        slotMoveNewTmp = f'{openFolderPath}/Presets/{selectedSlot}_tmp.bin'
##        os.rename(slotMoveOld, slotMoveNewTmp)
##        os.rename(slotMoveNew, slotMoveOld)
##        os.rename(slotMoveNewTmp, slotMoveNew)
##    else:
##        os.rename(slotMoveOld, slotMoveNew)
##    presetSlotsPopulate()
##    presetListbox.selection_clear(selectedSlot)
##    presetListbox.selection_set(selectedSlot-1)
##    loadSlot()

## moves slot down when called, unused  
##def moveFileDown():
##    if selectedSlot == presetCount-1:
##        return
##    if selectedSlot < 8:
##        slotMoveOld = f'{openFolderPath}/Presets/0{selectedSlot}.bin'
##        slotMoveNew = f'{openFolderPath}/Presets/0{selectedSlot+1}.bin'
##    else:
##        slotMoveOld = f'{openFolderPath}/Presets/{selectedSlot}.bin'
##        slotMoveNew = f'{openFolderPath}/Presets/{selectedSlot+1}.bin'   
##    if os.path.exists(slotMoveNew) == True:
##        if selectedSlot < 8:
##            slotMoveNewTmp = f'{openFolderPath}/Presets/0{selectedSlot+1}_tmp.bin'
##        else:
##            slotMoveNewTmp = f'{openFolderPath}/Presets/{selectedSlot+1}_tmp.bin'
##        os.rename(slotMoveOld, slotMoveNewTmp)
##        os.rename(slotMoveNew, slotMoveOld)
##        os.rename(slotMoveNewTmp, slotMoveNew)
##    else:
##        os.rename(slotMoveOld, slotMoveNew)
##    presetSlotsPopulate()
##    presetListbox.selection_clear(selectedSlot)
##    presetListbox.selection_set(selectedSlot+1)
##    loadSlot()

## function to change XNAME of a serialized preset file
def changeSerPresetXname(*args):
    xname = tk.StringVar()    
    newXname = tk.StringVar()
    frontend = tk.StringVar()
    newFrontend = tk.StringVar()
    pvehicle = tk.StringVar()
    newPvehicle = tk.StringVar()
    
    openSerPresetFile = filedialog.askopenfilename(title="Open your serialized preset", filetypes=[("NFSMW Binary preset", "*.bin *.BIN")])
    if openSerPresetFile == '':
            return
    
    with open (openSerPresetFile, "rb") as preset:
        if preset.read(3) != b'gMp':
            badFile = messagebox.showerror(title="Error", message="Not a serialized Binary preset .bin file")
            return
        preset.seek(0)
        openSerPreset=bytearray(preset.read())
        preset.seek(28)
        filesizeData=struct.unpack('h',preset.read(2))[0]
        presetChunk = openSerPreset[36:].split(b'\x00', 4)
        xname.set(presetChunk[1].decode())
        frontend.set(presetChunk[2].decode())
        pvehicle.set(presetChunk[3].decode())

    def chgSerPreXnameOkToggle(*args):
        if not newXname.get():
            chgSerPreXnameOkBtn.state(['disabled'])
            chgSerPreXnameTop.unbind('<Return>')
        else:
            chgSerPreXnameOkBtn.state(['!disabled'])
            chgSerPreXnameTop.bind('<Return>', chgSerPreXnameOk)
            
    def chgSerPreXnameOk(*args):
        with open (openSerPresetFile, "rb") as preset:
            openSerPreset=bytearray(preset.read())
            presetChunk[0] = presetChunk[0].replace(xname.get().encode('ascii'), newXname.get().encode('ascii'))+b'\x00'
            presetChunk[1] = newXname.get().encode('ascii')+b'\x00'
            presetChunk[2] = newFrontend.get().encode('ascii')+b'\x00'
            presetChunk[3] = newPvehicle.get().encode('ascii')+b'\x00'
            presetChunk[4] = presetChunk[4].replace(xname.get().encode('ascii'), newXname.get().encode('ascii'))
            savePreset = filedialog.asksaveasfilename(title="Save NFSMW serialized preset as...", filetypes=[("NFSU2 Binary preset", "*.bin *.BIN")], defaultextension=[".bin"])
            if savePreset == "":
                pass
            with open (savePreset, 'wb') as presetWrite:
                presetWrite.write(openSerPreset[0:36])
                presetWrite.write(presetChunk[0])
                presetWrite.write(presetChunk[1])
                presetWrite.write(presetChunk[2])
                presetWrite.write(presetChunk[3])
                presetWrite.write(presetChunk[4])
                if len(xname.get()) < len(newXname.get()):
                    preset.seek(28)
                    filesizeDataNew=struct.pack('h',filesizeData+200)
                    presetWrite.seek(28)
                    presetWrite.write(filesizeDataNew)

            presetWrite.close()                
            chgSerPreXnameTop.destroy()

    def chgSerPreXnameCancel(*args):
        newXname.set('')
        chgSerPreXnameTop.destroy()
        return

    chgSerPreXnameTop = tk.Toplevel(padx='5', pady='5')
    chgSerPreXnameTop.title("Change serialized preset XNAME")
    chgSerPreXnameTop.resizable(False,False)
    if os.name == "nt":
        chgSerPreXnameTop.attributes('-toolwindow',1)
    chgSerPreXnameTop.minsize(300,200)
    chgSerPreXnameTop.focus_force()
    chgSerPreXnameTop.grab_set()
    chgSerPreXnameCurLabel = ttk.Label(chgSerPreXnameTop, text=f'Current XNAME/MODEL: {xname.get()}')
    chgSerPreXnameCurLabel.grid(row=0, column=0, columnspan=3, sticky='NSEW')
    chgSerPreFeCurLabel = ttk.Label(chgSerPreXnameTop, text=f'Current frontend: {frontend.get()}')
    chgSerPreFeCurLabel.grid(row=1, column=0, columnspan=3, sticky='NSEW')
    chgSerPrePvehCurLabel = ttk.Label(chgSerPreXnameTop, text=f'Current pvehicle: {pvehicle.get()}')
    chgSerPrePvehCurLabel.grid(row=2, column=0, columnspan=3, sticky='NSEW')
    chgSerPreXnameSpacerLbl = ttk.Label(chgSerPreXnameTop, text="")
    chgSerPreXnameSpacerLbl.grid(row=3, column=0, sticky="NSEW", pady='5')
    chgSerPreXnameNewLbl = ttk.Label(chgSerPreXnameTop, text="New XNAME/MODEL:")
    chgSerPreXnameNewLbl.grid(row=4, column=0, columnspan=3, sticky='NSEW')
    chgSerPreXnameNewEnt = ttk.Entry(chgSerPreXnameTop, textvariable=newXname)
    chgSerPreXnameNewEnt.grid(row=5, column=0, columnspan=3, sticky='NSEW')
    chgSerPreXnameNewEnt.focus_set()
    chgSerPreFeNewLbl = ttk.Label(chgSerPreXnameTop, text="New frontend:")
    chgSerPreFeNewLbl.grid(row=6, column=0, columnspan=3, sticky='NSEW')
    chgSerPreFeNewEnt = ttk.Entry(chgSerPreXnameTop, textvariable=newFrontend)
    chgSerPreFeNewEnt.grid(row=7, column=0, columnspan=3, sticky='NSEW')
    chgSerPrePvehNewLbl = ttk.Label(chgSerPreXnameTop, text="New pvehicle:")
    chgSerPrePvehNewLbl.grid(row=8, column=0, columnspan=3, sticky='NSEW')
    chgSerPrePvehNewEnt = ttk.Entry(chgSerPreXnameTop, textvariable=newPvehicle)
    chgSerPrePvehNewEnt.grid(row=9, column=0, columnspan=3, sticky='NSEW')
    chgSerPreXnameOkBtn = ttk.Button(chgSerPreXnameTop, text="OK", command=chgSerPreXnameOk, state='disabled')
    chgSerPreXnameOkBtn.grid(row=10, column=1, pady='5', sticky='E')
    chgSerPreXnameCancBtn = ttk.Button(chgSerPreXnameTop, text="Cancel", command=chgSerPreXnameCancel)
    chgSerPreXnameCancBtn.grid(row=10, column=2, pady='5', sticky='E')

    chgSerPreXnameTop.columnconfigure(0, weight=1)

    regChgXname = chgSerPreXnameTop.register(inputCallback)
    chgSerPreXnameNewEnt.config(validate='key', validatecommand=(regChgXname, '%S','%P'))

    chgSerPreXnameTop.bind('<Escape>', chgSerPreXnameCancel)
    chgSerPreXnameTop.bind('<KeyRelease>', chgSerPreXnameOkToggle)
        
    root.wait_window(chgSerPreXnameTop)

## hacky keybind to open the Tools menu
def toolsMenuKbind(event):
    toolsMenuBtn.menu.post(toolsMenuBtn.winfo_rootx(),
                           toolsMenuBtn.winfo_rooty() + toolsMenuBtn.winfo_height())

## window elements    
openFilesFolderBtn = ttk.Button(topFrame, image=openBtnIcon, text='Open folder', style='TopButton.TButton', command=openFilesFolder)
openFilesFolderBtn.grid(row = 0, column=0, sticky='W')
openFilesFolderTip = Hovertip(openFilesFolderBtn, 'Open profile folder (Ctrl+O)')
reloadFilesFolderBtn = ttk.Button(topFrame, text='Reload folder', image=reloadBtnIcon, state='disabled', command=reloadFilesFolder)
reloadFilesFolderBtn.grid(row = 0, column=1, sticky='W')
reloadFilesFolderTip = Hovertip(reloadFilesFolderBtn, 'Reload folder (Ctrl+R)')
folderLabel = ttk.Label(topFrame, textvariable=folderPathStr)
folderLabel.grid(row = 0, column=2, sticky='W', padx=5)
topSpacerLbl = ttk.Label(topFrame, text='')
topSpacerLbl.grid(row = 0, column = 3, sticky = 'NSEW')

toolsMenuBtn = ttk.Menubutton(topFrame, text='Tools...', image=toolsBtnIcon, takefocus=True)
toolsMenuBtn.grid(row = 0, column=4, sticky="E")
toolsMenuBtnTip = Hovertip(toolsMenuBtn, 'Tools (Alt+T)')
toolsMenuBtn.menu = tk.Menu(toolsMenuBtn, tearoff=0)
toolsMenuBtn["menu"] = toolsMenuBtn.menu
toolsMenuBtn.menu.add_command(
    label="Change XNAME in serialized preset file...",
    accelerator="Ctrl-Shift-C",
    image=chgXnameBtnIcon,
    compound=tk.LEFT,
    command=changeSerPresetXname
    )
toolsMenuBtn.menu.add_separator()
toolsMenuBtn.menu.add_command(
    label="About...",
    accelerator="F1",
    image=aboutBtnIcon,
    compound=tk.LEFT,
    command=aboutDlg
    )
presetListbox = tk.Listbox(midFrameLeft, listvariable=presetSlotsVar, exportselection=False, state='disabled')
presetListbox.grid(row=0, column=0,sticky="NSEW")
presetLbScroll = ttk.Scrollbar(midFrameLeft, orient=tk.VERTICAL, command=presetListbox.yview)
presetLbScroll.grid(row=0, column=1, sticky="NSEW")
presetListbox.configure(yscrollcommand = presetLbScroll.set)
presetLbHScroll = ttk.Scrollbar(midFrameLeft, orient=tk.HORIZONTAL, command=presetListbox.xview)
presetLbHScroll.grid(row=1, column=0, sticky="NSEW")
presetListbox.configure(xscrollcommand = presetLbHScroll.set)
nameLbl = ttk.Label(midFrameRight, text="Enter XNAME for the preset", state = 'disabled')
nameLbl.grid(row=0, column=0, sticky="NSEW", pady='5')
nameEntry = ttk.Entry(midFrameRight, textvariable=nameText, state = 'disabled')
nameEntry.grid(row=1, column=0, columnspan=2, sticky="EW")
nameSaveBtn = ttk.Button(midFrameRight, text='Save slot', image=saveBtnIcon, command=saveSlot, compound='left', state = 'disabled', style='TabButton.TButton')
nameSaveBtn.grid(row=2, column=0 ,sticky="EW")
nameSaveTip = Hovertip(nameSaveBtn, 'Save preset (Ctrl+S)')
nameSaveAsBtn = ttk.Button(midFrameRight, text='Save slot as...', image=saveAsBtnIcon, compound='left', command=saveSlotAs, state = 'disabled', style='TabButton.TButton')
nameSaveAsBtn.grid(row=3, column=0 ,sticky="EW")
nameSaveAsTip = Hovertip(nameSaveAsBtn, 'Save slot as... (Ctrl+Shift+S)')
nameImportPresetBtn = ttk.Button(midFrameRight, text='Import slot', image=slotImportIcon, compound='left', command=importSlot, state = 'disabled', style='TabButton.TButton')
nameImportPresetBtn.grid(row=5, column=0 ,sticky="EW")
nameImportPresetTip = Hovertip(nameImportPresetBtn, 'Import preset (Ctrl+I)')

footSeparator = ttk.Separator(bottomFrame, orient="horizontal")
footSeparator.grid(row = 0, column = 0, sticky="EW")
statusLabel = ttk.Label(bottomFrame, textvariable=statusStr, width = 10)
statusLabel.grid(row = 1, column = 0, padx=2, pady=2, sticky='EW')

regNameEntry = root.register(inputCallback)
nameEntry.config(validate='key', validatecommand=(regNameEntry, '%S','%P'))

presetListbox.bind("<<ListboxSelect>>", loadSlot)

root.bind('<Control-o>', openFilesFolder)
root.bind('<Control-O>', openFilesFolder)
root.bind('<Control-s>', saveSlot)
root.bind('<Control-S>', saveSlot)
root.bind('<Control-Shift-s>', saveSlotAs)
root.bind('<Control-Shift-S>', saveSlotAs)
root.bind('<Control-r>', reloadFilesFolder) 
root.bind('<Control-R>', reloadFilesFolder)
root.bind('<Alt-t>', toolsMenuKbind)
root.bind('<Alt-T>', toolsMenuKbind)
root.bind('<Control-Shift-c>', changeSerPresetXname)
root.bind('<Control-Shift-C>', changeSerPresetXname)
root.bind('<F1>', aboutDlg)
root.bind('<F1>', aboutDlg)
root.bind('<Control-i>', importSlot)
root.bind('<Control-I>', importSlot)

root.mainloop()
